/* ram_drive.c */

#include <OS.h>
#include <Drivers.h>
#include <KernelExport.h>
#include <Mime.h>
#include <stdlib.h>


status_t vd_open(const char *name, uint32 flags, void **cookie);
status_t vd_free (void *cookie);
status_t vd_close(void *cookie);
status_t vd_control(void *cookie, uint32 msg, void *buf, size_t size);
status_t vd_read(void *cookie, off_t pos, void *buf, size_t *count);
status_t vd_write(void *cookie, off_t pos, const void *buf, size_t *count);

static void format_ram_drive(void* buf);
static uchar* create_ram_drive_area(size_t drive_size);
static status_t delete_ram_drive_area(void);
static void emulate_seek(off_t pos);


enum {
	RAM_DRIVE_RELEASE_MEMORY  = B_DEVICE_OP_CODES_END+1,
	RAM_DRIVE_EMULATE_SEEK    = B_DEVICE_OP_CODES_END+2
};

enum {
	RAM_DRIVE_SIZE        	= 8*1024*1024,	/* bytes */
	RAM_BLOCK_SIZE        	= 512,
	PREFETCH_BUFFER_SIZE  	= 32*1024,
	MAX_SEEK_TIME         	= 1000			/* microseconds */
};


static const char *vd_name[] = { "disk/virtual/ram_drive", NULL };
static const char *ram_drive_area_name = "RAM drive area";

static const uchar icon_disk[B_LARGE_ICON * B_LARGE_ICON];
static const uchar icon_disk_mini[B_MINI_ICON * B_MINI_ICON];

static int emulate_seek_flag = FALSE;
static uchar * ram = NULL;

static device_hooks vd_devices = {
	vd_open,
	vd_close,
	vd_free,
	vd_control,
	vd_read,
	vd_write,
	NULL,
	NULL,
	NULL,
	NULL
};



status_t
init_driver(void)
{
	dprintf("vd driver: %s %s, init_driver()\n",  __DATE__, __TIME__);
	ram = create_ram_drive_area(RAM_DRIVE_SIZE);
	if (ram == NULL)
		return B_ERROR;
	dprintf("vd driver: ram drive area at %p\n", ram);
	return B_NO_ERROR;
}

void
uninit_driver(void)
{
	dprintf("vd driver: uninit_driver()\n");
}

const char**
publish_devices()
{
	dprintf("vd driver: publish_devices()\n");
	return vd_name;
}

device_hooks*
find_device(const char* name)
{
	dprintf("vd driver: find_device()\n");
	return &vd_devices;
}

status_t
vd_open(const char *dname, uint32 flags, void **cookie)
{
	dprintf("vd driver: open(%s)\n", dname);
	return B_NO_ERROR;
}

status_t
vd_free (void *cookie)
{
	dprintf("vd driver: free()\n");
	return B_NO_ERROR;
}

status_t
vd_close(void *cookie)
{
	dprintf("vd driver: close()\n");
	return B_NO_ERROR;
}

status_t
vd_read(void *cookie, off_t pos, void *buf, size_t *count)
{
	size_t len;
	status_t ret = B_NO_ERROR;
	
	if (pos >= RAM_DRIVE_SIZE) {
		len = 0;
	} else {
		len = (pos + (*count) > RAM_DRIVE_SIZE) ?
		      (RAM_DRIVE_SIZE - pos) : (*count);
		emulate_seek(pos);
		memcpy(buf, ram+pos, len);
	}
	*count = len;
	return ret;
}

status_t
vd_write(void *cookie, off_t pos, const void *buf, size_t *count)
{
	size_t len;
	status_t ret = B_NO_ERROR;
	
	if (pos >= RAM_DRIVE_SIZE) {
		len = 0;
	} else {
		len = (pos + (*count) > RAM_DRIVE_SIZE) ?
		      (RAM_DRIVE_SIZE - pos) : (*count);
		emulate_seek(pos);
		memcpy(ram+pos, buf, len);
	}
	*count = len;
	return ret;
}

status_t
vd_control(void *cookie, uint32 ioctl, void *arg1, size_t len)
{
	device_geometry *dinfo;
	device_icon *dicon;
	
	switch (ioctl) {
	
	/* generic mass storage device IO control codes */
	
	case B_GET_GEOMETRY:
		dprintf("vd driver: control(B_GET_GEOMETRY)\n");
		
		dinfo = (device_geometry *) arg1;
		
		dinfo->bytes_per_sector = RAM_BLOCK_SIZE;
		dinfo->sectors_per_track = RAM_DRIVE_SIZE / RAM_BLOCK_SIZE;
		dinfo->cylinder_count = 1;
		dinfo->head_count = 1;
		dinfo->device_type = B_DISK;
		dinfo->removable = FALSE;
		dinfo->read_only = FALSE;
		dinfo->write_once = FALSE;

		return B_NO_ERROR;
	
	case B_FORMAT_DEVICE:
		dprintf("vd driver: control(B_FORMAT_DEVICE)\n");
		
		format_ram_drive(ram);
		return B_NO_ERROR;
	
	case B_GET_DEVICE_SIZE:
		dprintf("vd driver: control(B_GET_DEVICE_SIZE)\n");
		
		*(size_t*)arg1 = RAM_DRIVE_SIZE;
		return B_NO_ERROR;
	
	case B_GET_ICON:
		dprintf("vd driver: control(B_GET_ICON)\n");
		
		dicon = (device_icon *) arg1;

		switch (dicon->icon_size) {
		case B_LARGE_ICON:
			memcpy(dicon->icon_data, icon_disk, B_LARGE_ICON * B_LARGE_ICON);
			break;
		case B_MINI_ICON:
			memcpy(dicon->icon_data, icon_disk_mini, B_MINI_ICON * B_MINI_ICON);
			break;
		default:
			return B_BAD_TYPE;
		}
		return B_NO_ERROR;

	/* device specific IO control codes */
	
	case RAM_DRIVE_RELEASE_MEMORY:
		return delete_ram_drive_area();
	
	case RAM_DRIVE_EMULATE_SEEK:
		emulate_seek_flag = *(int*)arg1;
		return B_NO_ERROR;
	
	default:
		dprintf("vd driver: control(%ld) unknown\n", ioctl);
		return B_ERROR;
	}
}

static void
format_ram_drive(void* buf)
{
	static const char format_str[16] = "RAM drive       ";
	uchar* ptr = (uchar*)buf;
	off_t i;
	
	dprintf("vd driver: format_ram_drive(%p)\n", buf);
	for (i = 0; i < RAM_DRIVE_SIZE / 16; i++) {
		memcpy(ptr, format_str, 16);
		ptr += 16;
	}
}

static uchar*
create_ram_drive_area(size_t drive_size)
{
	void* addr;
	
	area_id area = find_area(ram_drive_area_name);
	if (area == B_NAME_NOT_FOUND) {
		area = create_area(ram_drive_area_name, &addr, B_ANY_KERNEL_ADDRESS,
			drive_size, B_LAZY_LOCK, B_READ_AREA | B_WRITE_AREA);
		if (area < B_OK)
			addr = NULL;
	} else {
		area_info info;
		get_area_info(area, &info);
		addr = info.address;
	}
	return (uchar *) addr;
}

static status_t
delete_ram_drive_area(void)
{
	area_id  area = find_area(ram_drive_area_name);
	if (area == B_NAME_NOT_FOUND)
		return B_ERROR;
	else
		return delete_area(area);
}

static void
emulate_seek(off_t pos)
{
	static off_t old_pos = 0;
	
	if (emulate_seek_flag) {
		if (abs(pos - old_pos) > PREFETCH_BUFFER_SIZE) {
			old_pos = pos;
			snooze((rand() * MAX_SEEK_TIME) / RAND_MAX);
		}
	}
}
